<?php
declare(strict_types=1);

/**
 * verify.php — JSON API صارم.
 * يقرأ: docType, docNumber, g-recaptcha-response
 * يعتمد على config.php لبيانات الاتصال بقاعدة البيانات، ومجلد الرفع، وسر الـ reCAPTCHA.
 * يرجع دائمًا JSON بالهيكل:
 * { ok:bool, msg:string, file?:string, fileExists?:bool, label?:string, searched?:{id?:int, path?:string} }
 */

@ini_set('display_errors','0');
@ini_set('zlib.output_compression','0');
if (function_exists('apache_setenv')) { @apache_setenv('no-gzip','1'); }
if (function_exists('header_remove')) {
  @header_remove('X-Powered-By');
  @header_remove('Server');
}
ob_start(); // نمسح أي مخرجات غير مقصودة
// هنطلع JSON فقط
header('Content-Type: application/json; charset=utf-8');

function jexit(array $arr, int $code = 200){
  // مسح أي مخرجات سابقة
  if (ob_get_length()) { @ob_end_clean(); }
  http_response_code($code);
  echo json_encode($arr, JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES);
  exit;
}

/**
 * قراءة قيمة من مصفوفة إعدادات مع دعم أسماء بديلة ومسارات متعشقة (dot-path).
 * مثال: cfg($config, 'db_host') سيجرّب ['db_host','host','hostname','db.host'] …إلخ
 */
function cfg(array $c, string $k, $default = null){
  $alts = [
    // اتصال قاعدة البيانات (ندعم المفاتيح المسطّحة والمتعشقة داخل db.*)
    'db_host'         => ['db_host','host','hostname','db.host'],
    'db_user'         => ['db_user','user','username','db.user'],
    'db_pass'         => ['db_pass','pass','password','db.pass'],
    'db_name'         => ['db_name','name','database','db.name'],
    'db_charset'      => ['db_charset','charset','db.charset'],

    // جدول البيانات
    'table'           => ['table','db_table','db.table'],

    // رفع الملفات
    'uploads_dir'     => ['uploads_dir','uploads','uploadDir'],

    // reCAPTCHA secret
    'recaptcha_secret'=> ['recaptcha_secret','recaptchaSecret','secret','security.recaptcha_secret']
  ];

  $candidates = $alts[$k] ?? [$k];

  foreach ($candidates as $path) {
    // دعم dot-path مثل: "db.host"
    $parts = explode('.', $path);
    $ref = $c; $found = true;
    foreach ($parts as $p) {
      if (is_array($ref) && array_key_exists($p, $ref)) {
        $ref = $ref[$p];
      } else {
        $found = false; break;
      }
    }
    if ($found && $ref !== '' && $ref !== null) {
      return $ref;
    }
  }
  return $default;
}

try {
  // 0) قراءة التهيئة
  $config = require __DIR__ . '/config.php';
  if (!is_array($config)) { jexit(['ok'=>false,'msg'=>'خطأ في ملف الإعدادات'], 500); }

  $uploadsDir = cfg($config, 'uploads_dir', __DIR__ . '/uploads');
  $tableName  = cfg($config, 'table', 'documents');

  // 1) قراءة المُدخلات
  $docType   = isset($_POST['docType']) ? trim((string)$_POST['docType']) : '';
  $docNumber = isset($_POST['docNumber']) ? trim((string)$_POST['docNumber']) : '';
  $token     = isset($_POST['g-recaptcha-response']) ? (string)$_POST['g-recaptcha-response'] : '';

  if ($docType === '' || $docNumber === '') {
    jexit(['ok'=>false,'msg'=>'الرجاء إدخال البيانات المطلوبة (نوع المستند ورقم المستند).'], 400);
  }

  // 2) reCAPTCHA (مع دعم تخطي التشخيص)
  $bypass = false;
  if (!empty($_SERVER['HTTP_X_BYPASS_RECAPTCHA']) && $_SERVER['HTTP_X_BYPASS_RECAPTCHA'] === '1') {
    $bypass = true;
  }

  if (!$bypass) {
    $secret = cfg($config, 'recaptcha_secret', getenv('RECAPTCHA_SECRET') ?: '');
    if ($secret === '') {
      jexit(['ok'=>false,'msg'=>'لم يتم ضبط مفاتيح التحقق.'], 500);
    }

    $payload = http_build_query(['secret'=>$secret,'response'=>$token,'remoteip'=>$_SERVER['REMOTE_ADDR'] ?? '']);
    $verifyURL = 'https://www.google.com/recaptcha/api/siteverify';

    $resp = null; $code = 0;
    if (function_exists('curl_init')) {
      $ch = curl_init($verifyURL);
      curl_setopt_array($ch, [
        CURLOPT_POST=>true, CURLOPT_POSTFIELDS=>$payload,
        CURLOPT_RETURNTRANSFER=>true, CURLOPT_TIMEOUT=>10,
      ]);
      $resp = curl_exec($ch);
      $code = (int)curl_getinfo($ch, CURLINFO_RESPONSE_CODE);
      curl_close($ch);
    } else {
      $ctx = stream_context_create([
        'http'=>[
          'method'=>'POST',
          'header'=>"Content-Type: application/x-www-form-urlencoded\r\n".
                    "Content-Length: ".strlen($payload)."\r\n",
          'content'=>$payload,
          'timeout'=>10,
        ]
      ]);
      $resp = @file_get_contents($verifyURL, false, $ctx);
      $code = 200; // best effort
    }

    $okCaptcha = false;
    if ($resp) {
      $j = json_decode($resp, true);
      $okCaptcha = is_array($j) && !empty($j['success']);
    }
    if (!$okCaptcha) {
      jexit(['ok'=>false,'msg'=>'فشل التحقق أنا لست روبوت.'], 403);
    }
  }

  // 3) الاتصال بقاعدة البيانات
  $host = cfg($config,'db_host','localhost');
  $user = cfg($config,'db_user','');
  $pass = cfg($config,'db_pass','');
  $name = cfg($config,'db_name','');
  $charset = cfg($config,'db_charset','utf8mb4');

  if ($user==='' || $name==='') {
    jexit(['ok'=>false,'msg'=>'إعدادات قاعدة البيانات غير مكتملة.'], 500);
  }

  mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);
  $mysqli = new mysqli($host, $user, $pass, $name);
  $mysqli->set_charset($charset);

  // 4) الاستعلام المُحضّر
  // الجدول يفترض أعمدة: docType (VARCHAR), docNumber (VARCHAR), id (INT/PK)
  $sql = "SELECT id FROM `$tableName` WHERE `docType` = ? AND `docNumber` = ? LIMIT 1";
  $stmt = $mysqli->prepare($sql);
  $stmt->bind_param('ss', $docType, $docNumber);
  $stmt->execute();
  $res = $stmt->get_result();
  $row = $res->fetch_assoc();

  if (!$row) {
    jexit(['ok'=>false,'msg'=>'البيانات لا تطابق قاعدة البيانات'], 200);
  }

  $id = (int)$row['id'];
  // 5) تكوين المسار: uploads/{id}.pdf
  $uploadsReal = realpath($uploadsDir) ?: $uploadsDir;
  $fileRel = "uploads/{$id}.pdf"; // للواجهة
  $fileAbs = $uploadsReal . DIRECTORY_SEPARATOR . "{$id}.pdf";

  $exists = is_file($fileAbs);

  $label = $docType . '_' . $docNumber . ' المستندات المتاحة';

  jexit([
    'ok'         => true,
    'msg'        => 'تم التحقق',
    'label'      => $label,
    'file'       => $fileRel,
    'fileExists' => (bool)$exists,
    'searched'   => ['id'=>$id, 'path'=>$fileRel]
  ], 200);

} catch (Throwable $e) {
  // لا نُظهر تفاصيل حساسة للواجهة
  jexit(['ok'=>false,'msg'=>'خطأ داخلي في الخادم'], 500);
}
